#!/bin/bash

# ==========================================================
# Secure Messaging Gateway Initialization Script
# Version: 1.0.9
# Build Date: 2025-09-08
# ==========================================================

ensure_repo_correct() {
    local ALIAS="$1"
    local EXPECTED_URL="$2"

    local CURRENT_URL
    CURRENT_URL=$(zypper lr -u | awk -v alias="$ALIAS" '
        $0 ~ alias { getline; print $3 }')

    if [[ "$CURRENT_URL" == "$EXPECTED_URL" ]]; then
        echo "Repo '$ALIAS' already has the correct URL, no changes made."
    else
        echo "Repo '$ALIAS' exists with outdated URL or does not exist, fixing..."
        zypper rr "$ALIAS" 2>/dev/null
        zypper --gpg-auto-import-keys ar "$EXPECTED_URL" "$ALIAS"
        zypper --gpg-auto-import-keys ref
    fi
}

if ! [ -f "Secure-Messaging-Gateway-EULA.txt" ]; then
        echo ""
        echo "EULA is missing.  Please follow the installation instructions to use this installer."
        echo ""
        exit
fi

clear

# Extract OS and VERSION from /etc/os-release
OS_ID=$(grep -i "^ID=" /etc/os-release | cut -d= -f2 | tr -d '"')
VERSION_ID=$(grep -i "^VERSION_ID=" /etc/os-release | cut -d= -f2 | tr -d '"')

# Normalize to lowercase
OS_ID=${OS_ID,,}
VERSION_ID=${VERSION_ID,,}

# Build REGISTEROS
case "$OS_ID" in
  sles)
    REGISTEROS="sles-${VERSION_ID//./-}-x86_64"
    ;;
  opensuse-leap)
    REGISTEROS="opensuse-leap-${VERSION_ID//./-}-x86_64"
    ;;
  *)
    REGISTEROS=""
    ;;
esac

# Build REGISTEROS
case "$OS_ID" in
  sles)
    REGISTEROS="sles-${VERSION_ID//./-}-x86_64"
    ;;
  opensuse-leap)
    REGISTEROS="opensuse-leap-${VERSION_ID//./-}-x86_64"
    ;;
  *)
    REGISTEROS=""
    ;;
esac

# Assign OPSYS number based on REGISTEROS
case "$REGISTEROS" in
  sles-15-5-x86_64) OPSYS=1 ;;
  sles-15-6-x86_64) OPSYS=4 ;;
  sles-15-7-x86_64) OPSYS=5 ;;
  opensuse-leap-15-5-x86_64) OPSYS=2 ;;
  opensuse-leap-15-6-x86_64) OPSYS=3 ;;
  *) OPSYS=0 ;;
esac

## echo "DEBUG: OS_TARGET='$OS_TARGET'"

# Display detected OS
if [[ $OPSYS -eq 0 ]]; then
  echo "An unsupported OS was detected. Please check the documentation..."
else
  echo "OS detected: $REGISTEROS"
fi

# Refresh repositories after module activation
zypper --gpg-auto-import-keys ref

# On SLES 15 SP6/SP7, auto-register the SUSE Package Hub (“backports”) extension
if [[ $OPSYS -eq 4 || $OPSYS -eq 5 ]]; then
  SP_VERSION="${VERSION_ID}"    # use full version (e.g. "15.6")
  echo "Registering SUSE Package Hub for SLE 15-SP${SP_VERSION}…"
  SUSEConnect -p PackageHub/${SP_VERSION}/x86_64 || echo "WARNING: could not register PackageHub, continuing"
  zypper --gpg-auto-import-keys ref
fi

ShowSyntax() {
echo ""
echo "Usage:"
echo "smg-init.sh [email=registration email] [regcode-smg=registration code acquired from SLD] [--help] [-h]"
echo ""
}

# Read configuration parameters from the command line
for arg in $@
do

        if [[ ${arg,,} =~ ^email=.* ]]; then
                EMAIL=${arg:6}
        fi

        if [[ ${arg,,} =~ ^regcode-smg=.* ]]; then
                REGCODE=${arg:12}
        fi

        if [[ ${arg,,} =~ --help ]]; then
                ShowSyntax
                                exit
        fi

        if [[ ${arg,,} =~ -h ]]; then
                ShowSyntax
                                exit
        fi

done


echo "Welcome to the Secure Messaging Gateway installation"
echo ""
echo "Install script version: 1.0.8"
echo ""
echo "Before we begin, you need to read and accept the EULA"
echo "that will be presented to you. Navigate the EULA with"
echo "arrow keys and page up/down keys."
echo ""
echo "Once you have finished reading the EULA, press 'Q'."
echo ""
read -p "Press enter now to display the EULA "

less Secure-Messaging-Gateway-EULA.txt

clear

echo "Welcome to the Secure Messaging Gateway installation"
echo ""
read -p "Enter 'A' to accept the EULA, or any other key to exit: " EULAACCEPT

case $EULAACCEPT in
    [Aa] )  ;;
    * )
        echo ""
        echo "EULA Rejected, exiting"
        exit
        ;;
esac

clear

echo "This installer will perform most of the installation tasks for your"
echo "initial Secure Messaging Gateway (SMG) installation"
echo ""
echo "This process will install the base SMG files and its dependencies."
echo "On completion you will then be able to begin the SMG application"
echo "configuration, as described in the product documentation."
echo ""
echo "Please make sure you have read the installation reference guide prior"
echo "to running this script to ensure a smooth installation process."
echo "If prerequisite requirements have not been met, this script may abort"
echo "and leave your system in a semi-installed state.  It is recommended"
echo "that you start the install from scratch if the installation fails"
echo "part way through."
echo ""


echo ""

read -p "Do you want to proceed with the installation process? (Y/n) " yn

case $yn in
                [nN] )  echo "Installation aborted";
                                exit 1;
esac

CFG_SMG=1
CFG_FIREWALL=1
CFG_APACHE=1
CFG_POSTGRES=1
CFG_POSTGRES_PASS=1
CFG_POSTFIX=1
CFG_START_SERVICES=1
CFG_OPENDKIM=1
POSTGRES_PASSWORD=postgres

EchoOption() {
        OPTION_ENABLED=" "
        if [[ $1 -eq 1 ]]; then
                OPTION_ENABLED="X"
        fi
        echo "[$OPTION_ENABLED] $2 - $3"
}

ShowCurrentOptions() {

EchoOption "$CFG_SMG" "S" "Install the Secure Messaging Gateway package"

if [ -d "/opt/opentext/smg" ] || [ $CFG_SMG -eq 1 ]; then
        EchoOption "$CFG_FIREWALL" "F" "Configure firewall"
        EchoOption "$CFG_START_SERVICES" "V" "Enable service dependencies"
        EchoOption "$CFG_APACHE" "A" "Configure Apache web server"
        EchoOption "$CFG_OPENDKIM" "D" "Configure OpenDKIM service"
        EchoOption "$CFG_POSTFIX" "X" "Disable Postfix email server"
        EchoOption "$CFG_POSTGRES" "P" "Configure Postgres database server"
        EchoOption "$CFG_POSTGRES_PASS" "K" "Set Postgres user password"
else
        echo ""
        echo "System configuration options can only be selected if SMG is either"
        echo "chosen as an option, or already installed on this system"
        echo ""
fi

echo ""
if [ -d "/opt/opentext/smg" ] || [ $CFG_SMG -eq 1 ]; then
        echo "B - Begin setup"
fi
echo "Q - Quit setup"
if [ -d "/opt/opentext/smg" ] || [ $CFG_SMG -eq 1 ]; then
        echo "? - Show configuration process details"
fi
}

ToggleValue() {
        if [ $1 == 1 ]; then
                return 0
        fi
        return 1
}

ShowPlannedChanges() {
        echo ""
        if [[ $CFG_SMG -eq 1 ]]; then
                echo "* Install Secure Messaging Gateway *"
                echo "The installation of SMG will be run"
                echo ""
        fi

        if [[ $CFG_FIREWALL -eq 1 ]]; then
                echo "* Firewall configuration *"
                echo "Ports will be enabled to allow public access for smtp, http, postgres, dns, ClamAV, OpenDKIM smg communications (4928-4930,4932)"
                echo ""
        fi

        if [[ $CFG_START_SERVICES -eq 1 ]]; then
                echo "* Service dependencies *"
                echo "System services that SMG requires will be enabled - apache2 opendkim postgresql"
                echo ""
        fi

        if [[ $CFG_APACHE -eq 1 ]]; then
                echo "* Apache web server configuration *"
                echo "The web server will be configured to listen on port 80 and 443 for IPv4"
                echo ""
        fi

        if [[ $CFG_POSTFIX -eq 1 ]]; then
                echo "* Postfix configuration *"
                echo "The Postfix service will be disabled to prevent conflicts with SMG smtp interfaces"
                echo ""
        fi

        if [[ $CFG_POSTGRES -eq 1 ]]; then
                echo "* Postgres database configuration *"
                echo "Ports and security settings will be applied to the Postgres database to allow access by SMG"
                echo ""
        fi

        if [[ $CFG_POSTGRES_PASS -eq 1 ]]; then
                echo "* Postgres database master password *"
                echo "The postgres database user will be set, allowing SMG to automatically connect and manage its databases"
                echo ""
        fi
        if [[ $CFG_OPENDKIM -eq 1 ]]; then
                echo "* Modify OpenDKIM configuration *"
                echo "The OpenDKIM configuration file will be adjusted to allow access and synchronization with the SMG application and admin UI"
                echo ""
        fi
}

VALID_OPTION=1
NOCLEAR=0

while true; do

if [[ $NOCLEAR -eq 0 ]]; then
clear
fi

NOCLEAR=0

echo "Secure Messaging Gateway installation preparation tasks"
echo ""
echo "The listed items will be performed on your server configuration"
echo "during the initialization process.  If you plan to manually"
echo "configure any of the additional items, disable them now."
echo "You may re-run this script at any time to selectively run any"
echo "of the pre-installation tasks."
echo ""
echo "Toggle configuration options by pressing the letter at"
echo "the left of a line."
echo ""

if [[ $VALID_OPTION -eq 0 ]]; then
        echo "Invalid selection.  Please select from the available options, or Q to quit."
        echo ""
        VALID_OPTION=1
fi

        ShowCurrentOptions
        echo ""
        read -p "Enter your selection, followed by enter: " USERINPUT
        case $USERINPUT in
                [Ss] )  ToggleValue $CFG_SMG;CFG_SMG=$?;;
                [Ff] )  ToggleValue $CFG_FIREWALL;CFG_FIREWALL=$?;;
                [Vv] )  ToggleValue $CFG_START_SERVICES;CFG_START_SERVICES=$?;;
                [Aa] )  ToggleValue $CFG_APACHE;CFG_APACHE=$?;;
                [Dd] )  ToggleValue $CFG_OPENDKIM;CFG_OPENDKIM=$?;;
                [Xx] )  ToggleValue $CFG_POSTFIX;CFG_POSTFIX=$?;;
                [Pp] )  ToggleValue $CFG_POSTGRES;CFG_POSTGRES=$?;;
                [Kk] )  ToggleValue $CFG_POSTGRES_PASS;CFG_POSTGRES_PASS=$?;;
                [?] )   clear
                                if [ -d "/opt/opentext/smg" ] || [ $CFG_SMG -eq 1 ]; then
                                        ShowPlannedChanges
                                fi
                                NOCLEAR=1;;
                [Bb] )
                                if [ -d "/opt/opentext/smg" ] || [ $CFG_SMG -eq 1 ]; then
                                        break
                                fi
                                ;;

                [Qq] )  exit;;
        * )
                                VALID_OPTION=0
                                ;;
        esac


done

 # Register required modules for SLES 15 SP6/SP7
 if [[ "$REGISTEROS" == "sles-15-6-x86_64" || "$REGISTEROS" == "sles-15-7-x86_64" ]]; then
   # Use the real VERSION_ID ("15.6" or "15.7"), not a substring of REGISTEROS
   SP_VERSION="${VERSION_ID}"

   echo "Registering required SUSE modules for SLES ${SP_VERSION}…"

  sudo SUSEConnect -p sle-module-desktop-applications/${SP_VERSION}/x86_64 || true
  sudo SUSEConnect -p sle-module-development-tools/${SP_VERSION}/x86_64 || true
  sudo SUSEConnect -p sle-module-web-scripting/${SP_VERSION}/x86_64 || true
  sudo SUSEConnect -p sle-module-systems-management/${SP_VERSION}/x86_64 || true

  # **Automatically** pull in the Package Hub (“backports”) repo**
  sudo SUSEConnect -p PackageHub/${SP_VERSION}/x86_64 || true

 # now refresh everything…
 zypper --gpg-auto-import-keys ref

 fi

 # now refresh everything…
 zypper --gpg-auto-import-keys ref

 # Now pull in the Package Hub (“backports”) repo for SP6/SP7
 if [[ $OPSYS -eq 4 || $OPSYS -eq 5 ]]; then
   SP_VERSION="${VERSION_ID}"     # use full version (e.g. "15.6")
   echo "Registering SUSE Package Hub for SLE 15-SP${SP_VERSION}…"
   SUSEConnect -p PackageHub/${SP_VERSION}/x86_64 || true
   zypper --gpg-auto-import-keys ref

   # sanity-check that the Package Hub repo really showed up
        if ! zypper lr | grep -iqE 'Package[- ]?Hub'; then
          echo
          echo "ERROR: SUSE Package Hub repo not enabled (looking for \"Package[- ]?Hub\")."
          echo "Please install the 'SUSE Package Hub' extension and re-run this script."
          exit 1
        fi

 fi

if [[ $CFG_SMG -eq 1 ]]; then

        echo ""

        if [ -z ${EMAIL+x} ]; then
                read -p "Enter the email address for registering your SMG product: " EMAIL
        fi
        if [ -z ${REGCODE+x} ]; then
                read -p "Enter your SMG product registration code: " REGCODE
        fi

        while true; do

                echo ""
                echo "Secure Messaging Gateway will be registered with the following information:"
                echo "Email address:     $EMAIL"
                echo "Registration code: $REGCODE"
                echo ""

                read -p "Are the registration credentials correct (y/N/q)? " CONFIRMREG
                case $CONFIRMREG in
                        [Yy] )  break;;
                        [Qq] ) exit;;
                esac

                echo ""
                echo "Enter updated registration details (or enter to keep existing value)"
                echo ""

                NEWEMAIL=""
                read -p "Enter the email address for registering your SMG product [$EMAIL]: " NEWEMAIL
                if [[ $NEWEMAIL != "" ]]; then
                        EMAIL=$NEWEMAIL
                fi
                NEWREGCODE=""
                read -p "Enter your SMG product registration code [$REGCODE]: " NEWREGCODE
                if [[ $NEWREGCODE != "" ]]; then
                        REGCODE=$NEWREGCODE
                fi

        done

fi

if [[ $CFG_POSTGRES_PASS -eq 1 ]]; then
        echo ""
        echo "You have selected to set the password for the Postgres database user.  Your selection for this"
        echo "change is important to get right."
        echo ""
        echo "If this server is being configured specifically to run SMG, then you should select the default"
        echo "option, which will set the password to 'postgres'.  During the web-based SMG setup wizard the"
        echo "default password of 'postgres' will be used."
        echo ""
        echo "If you want to set your own password it is important that this is not lost.  During the web-based"
        echo "SMG setup wizard you will need to enter the password that you set here into each of the input tabs"
        echo "for the database connections."

        PGPASSMETHOD=0
        PGPASSMETHOD_ATTEMPTS=0
        while [[ $PGPASSMETHOD -eq 0 ]]; do
                        echo ""
                        echo "Do you want to set the password to the default of 'postgres' [d] or enter a new password [n]?"
                        echo "Enter [q] to quit the installer"
                        echo
                        read -p "Enter selection (d/n): " PGPASSMETHOD

                        case $PGPASSMETHOD in
                                [dD] )  POSTGRES_PASSWORD="postgres"
                                                break;;

                                [nN] )  PGPASSMETHOD="n"
                                                break;;

                                [qQ] )  exit;;

                                * )     ((PGPASSMETHOD_ATTEMPTS++))
                                                echo ""
                                                echo "Invalid option.  Please select from the options presented"
                                                if [[ PGPASSMETHOD_ATTEMPTS -eq 2 ]]; then
                                                                echo ""
                                                                echo "There are only three valid options, it's not that hard"
                                                fi
                                                if [[ PGPASSMETHOD_ATTEMPTS -eq 5 ]]; then
                                                                echo ""
                                                                echo "You really should read the instructions before pressing random keys"
                                                fi
                                                if [[ PGPASSMETHOD_ATTEMPTS -eq 10 ]]; then
                                                                echo ""
                                                                echo "Please ask someone who is more competent to install this application, aborting to prevent probable damage"
                                                                exit
                                                fi
                                                ;;

                        esac
        done


        if [ $PGPASSMETHOD = "n" ]; then
                        echo ""

                        PASSWORDMATCH=0
                        while [[ $PASSWORDMATCH -eq 0 ]]; do
                                        PASSWORDMATCH=1
                                        echo "Enter new password"
                                        read -s NewPgPass
                                        if [ "$NewPgPass" = "" ]; then
                                                        PASSWORDMATCH=0
                                                        echo "The password cannot be empty"
                                                        continue
                                        fi
                                        echo "Re-enter password"
                                        read -s NewPgPassVerify
                                        if [ "$NewPgPass" != "$NewPgPassVerify" ]; then
                                                        echo "Passwords do not match, please try again"
                                                        PASSWORDMATCH=0
                                        else
                                                        POSTGRES_PASSWORD=$NewPgPass
                                        fi
                        done
        fi
fi

echo ""
echo "Proceeding with selected configuration options"
echo ""

remove_repo_by_url() {
    local TARGET_URL="$1"
    local MATCHING_ALIAS

    MATCHING_ALIAS=$(zypper lr -u | awk -v url="$TARGET_URL" '$0 ~ url {getline; print $1}')

    if [[ -n "$MATCHING_ALIAS" ]]; then
        echo "Removing repo with outdated URL: $TARGET_URL (alias: $MATCHING_ALIAS)"
        zypper rr "$MATCHING_ALIAS"
    fi
}

# Remove outdated Perl repo (15.5 style)
remove_repo_by_url "https://download.opensuse.org/repositories/devel:languages:perl/15.5"

# Remove outdated Editors repo (15.5 style)
remove_repo_by_url "https://download.opensuse.org/repositories/editors/15.5"

remove_repo_if_exists() {
  local ALIAS="$1"
  if zypper lr | grep -q "^.*| $ALIAS[[:space:]]"; then
    echo "Removing existing repository: $ALIAS"
    zypper rr "$ALIAS"
  fi
}

                #
                # —— 1) editor repo —— works for SLE/Leap 15.SP5,6,7 alike
                ensure_repo_correct "editors_15_6" \
                    "https://download.opensuse.org/repositories/editors/15.6/"

                # —— 2) separate Perl repo per SP —— SLE_15_SP5|6|7 (and reuse on Leap SP5/6/7)
                # strip off the “15.” prefix to get just “5”/“6”/“7”
                SP_NUM="${VERSION_ID#*.}"
                PERL_ALIAS="devel_languages_perl_SP${SP_NUM}"
                PERL_URL="https://download.opensuse.org/repositories/devel:/languages:/perl/SLE_15_SP${SP_NUM}/"

                # drop any old perl repo by that alias, then add the correct one
                zypper --non-interactive rr "${PERL_ALIAS}" 2>/dev/null || true
                ensure_repo_correct "${PERL_ALIAS}" "${PERL_URL}"

                # —— 3) final refresh before installing Perl modules
                zypper --gpg-auto-import-keys --non-interactive ref
                zypper in -y perl-WWW-Curl perl-XML-Writer nano make

        # Install suse_register for smg

        tar -xf smg-suseRegister-1.4.1.tar.bz2
        cd smg-suseRegister-1.4.1
        make install
	echo "suse_register_smg --restore-repos -a email=$EMAIL -a regcode-smg=$REGCODE -a ostarget=$REGISTEROS -a baseversion=24.3 -L /var/log/suse_register-smg.log"
        suse_register_smg --restore-repos -a email=$EMAIL -a regcode-smg=$REGCODE -a ostarget=$REGISTEROS -a baseversion=24.3 -L /var/log/suse_register-smg.log

        # Detect OS and Service Pack
        OS_ID=$(grep ^ID= /etc/os-release | cut -d= -f2 | tr -d '"')
        OS_VERSION_ID=$(grep ^VERSION_ID= /etc/os-release | cut -d= -f2 | tr -d '"')
        OS_SP=$(echo "$OS_VERSION_ID" | cut -d. -f2)

        # Determine OS target in correct SMG repo format
        if [[ "$OS_ID" == "sles" ]]; then
                OS_TARGET="SLE-15.${OS_SP}"
                REPO_DIR="SMG-2024-${OS_TARGET}-BASE-Product"
		DISTRIBUTION_TYPE="sle-15"
        elif [[ "$OS_ID" == "opensuse-leap" ]]; then
                OS_TARGET="LEAP-15.${OS_SP}"
                REPO_DIR="SMG-2024-${OS_TARGET}-BASE-Product"
		DISTRIBUTION_TYPE="opensuse-leap-15"
        else
                echo "Unsupported OS: $OS_ID"
                exit 1
        fi

if [[ $CFG_SMG -eq 1 ]]; then

#
# Purge any stale SMG repo/credentials so we start clean
#
zypper --non-interactive rr "SMG-2024-${OS_TARGET}-BASE-Product" 2>/dev/null || true
rm -f /etc/zypp/credentials.d/NCCcredentials

#
# Register (and retry if “needinfo” appears)
#
OUT=$(mktemp)
suse_register_smg --restore-repos \
  -a email="$EMAIL" \
  -a regcode-smg="$REGCODE" \
  -a ostarget="$REGISTEROS" \
  -a baseversion=24.3 \
  -L /var/log/suse_register-smg.log \
  2>&1 | tee "$OUT"

if grep -q '<needinfo' "$OUT"; then
  SECRET=$(grep '<param id="secret"' "$OUT" \
             | sed -e 's/.*<param id="secret">\([^<]*\)<\/param>.*/\1/')
  if [[ -n "$SECRET" ]]; then
    echo "↻ Re-submitting registration with secret from portal…"
    suse_register_smg --restore-repos \
      -a email="$EMAIL" \
      -a regcode-smg="$REGCODE" \
      -a secret="$SECRET" \
      -a ostarget="$REGISTEROS" \
      -a baseversion=24.3 \
      -L /var/log/suse_register-smg.log
  else
    echo "ERROR: Could not extract secret—registration failed."
    exit 1
  fi
fi
rm -f "$OUT"

if [[ ! -s /etc/zypp/credentials.d/NCCcredentials ]]; then
  echo "ERROR: NCCcredentials not created—see /var/log/suse_register-smg.log"
  exit 1
fi

#
# Build the proper SMG repo URL:
REPO_NAME="SMG-2024-${OS_TARGET}-BASE-Product"
REPO_URL="https://nu.novell.com/repo/\$RCE/${REPO_DIR}/${DISTRIBUTION_TYPE}-x86_64?credentials=NCCcredentials"

# (Re)register that repo once—only *after* you have fresh creds  
zypper --non-interactive rr "$REPO_NAME"        2>/dev/null || true  
echo "*** repo url - $REPO_URL $REPO_NAME"
#REPO_URL="https://nu.novell.com/repo/\$RCE/SMG-2024-LEAP-15.6-BASE-Product/opensuse-leap-15-x86_64?credentials=NCCcredentials"
zypper --gpg-auto-import-keys ar -f "$REPO_URL" "$REPO_NAME"  
zypper --gpg-auto-import-keys ref "$REPO_NAME"  

#
# Install SMG
#
# Testing the smg install step. This was remarked
zypper in -y smg

fi

        # Check if repo already exists
        if ! zypper lr | grep -q "$REPO_NAME"; then
                echo "Adding missing SMG repo for $OS_TARGET"

		echo "*** repo url1 - $REPO_URL $REPO_NAME"
                zypper --gpg-auto-import-keys ar -f "$REPO_URL" "$REPO_NAME"
                if [[ $? -ne 0 ]]; then
                        echo "Failed to add SMG repo: $REPO_URL"
                        exit 1
                fi
        fi

        # Refresh all repos to make sure it's usable
        zypper --gpg-auto-import-keys ref

        # Only add if not already present
        if ! zypper lr | grep -q "$REPO_NAME"; then
    echo "Manually adding missing SMG repo: $REPO_NAME"
    echo "*** repo url3 - $REPO_URL $REPO_NAME"
    zypper --gpg-auto-import-keys ar -f "$REPO_URL" "$REPO_NAME"
        fi


        if [[ $? -eq 99 ]]; then
                echo ""
                echo "SMG installation failed due to invalid registration details."
                echo "Check the log file for additional info: /var/log/suse_register-smg.log"
                exit
        fi

        if [[ $? -ne 0 ]]; then
                echo ""
                echo "SMG installation failed during registration."
                echo "Check the log file for additional info: /var/log/suse_register-smg.log"
                exit
        fi

        cd ..

        # Start smg install
        zypper --gpg-auto-import-keys ref

        INSTALLFAIL=1

        # Sometimes the repository is not enabled properly, retry 3 times before failing
        for i in {1..3}
        do
                # On LEAP, enable the repository that for some reason disables itself on first install attempt
                if [[ $OPSYS -eq 1 ]] || [[ $OPSYS -eq 2 ]] || [[ $OPSYS -eq 3 ]] || [[ $OPSYS -eq 4 ]]; then
                        zypp_dir=/etc/zypp/repos.d
                        for zypp_entry in "$zypp_dir"/*.repo
                        do
                                if [ -f "$zypp_entry" ] && [[ $zypp_entry == *"/nu_novell_com:SMG-"* ]]; then
                                echo Enabling SMG product repository
                                zypper mr --enable "$(basename ${zypp_entry%.*})"
                                fi
                        done
                fi

###                zypper in -y smg

                # Stop retrying if SMG installs correctly
                if [[ $? -eq 0 ]]; then
                        INSTALLFAIL=0
                        break;
                fi

        done

        # Check that smg eventually installed after up to three retries
        if [[ $INSTALLFAIL -ne 0 ]]; then
                echo ""
                echo "SMG RPM installation failed, check system logs and registration process log to troubleshoot this situation"
                echo "Registration log is located at: /var/log/suse_register-smg.log"
                exit
        fi

if [[ $CFG_START_SERVICES -eq 1 ]]; then
        # Enable all of the services required by SMG
        systemctl enable --now apache2 opendkim postgresql
fi

if [[ $CFG_SMG -eq 1 ]]; then
        # Enable the SMG service
        systemctl enable --now smg
fi

if [[ $CFG_APACHE -eq 1 ]]; then
        # Adjust the listen command in the apache config file to ensure it works for the SMG UI

        # If there is no uncommented Listen entry for port 80 add it
        grep -iEq "^\\s*listen\\s+80(\\W|$)" /etc/apache2/listen.conf
        if [[ $? -ne 0 ]]; then
                echo "Listen 80" >> /etc/apache2/listen.conf
        fi

        # If there is an indented, uncommented Listen entry for port 443, it is more than likely inside the defunct
        # SSL ifdef tags which don't work.  Comment this one out
        sed -i -E "s/(^\\s+Listen\\s+443.*)/#\\1/i" /etc/apache2/listen.conf

        # If there is no uncommented Listen entry for port 443, add it
        grep -iEq "^\\s*listen\\s+443(\\W|$)" /etc/apache2/listen.conf
        if [[ $? -ne 0 ]]; then
                echo "Listen 443" >> /etc/apache2/listen.conf
        fi

        systemctl restart apache2
fi

#
# Add a line to a configuration file if it is not located
# $1 - File to inspect
# $2 - Pattern to detect existing line
# $3 - Line to append if $2 is not located
# $4 - Optional pattern to comment out a line if the detect pattern was not located (i.e. comment out incompatible versions of the new line)
#
AddIfMissing() {
        grep -iEq $2 $1
        if [[ $? -ne 0 ]]; then
                if [ $4 != "" ]; then
                        sed -i -E "s/($4)/#\\1/i" $1
                fi
                echo $3 >> $1
        fi
}

if [[ $CFG_OPENDKIM -eq 1 ]]; then
        # Adjust the OpenDKIM configuration file to listen on an IP socket and link to SMG managed signing tables

        AddIfMissing "/etc/opendkim/opendkim.conf" "^SOCKET\\s+inet:4932@localhost" "SOCKET                  inet:4932@localhost" "^SOCKET\\s+.*"
        AddIfMissing "/etc/opendkim/opendkim.conf" "^Canonicalization\\s+relaxed/relaxed" "Canonicalization        relaxed/relaxed" "^Canonicalization\\s+.*"
        AddIfMissing "/etc/opendkim/opendkim.conf" "^Mode\\s+sv" "Mode                    sv" "^Mode\\s+.*"
        AddIfMissing "/etc/opendkim/opendkim.conf" "^KeyTable\\s+refile:/etc/opendkim/key.table" "KeyTable                refile:/etc/opendkim/key.table" "^KeyTable\\s+.*"
        AddIfMissing "/etc/opendkim/opendkim.conf" "^SigningTable\\s+refile:/etc/opendkim/signing.table" "SigningTable            refile:/etc/opendkim/signing.table" "^SigningTable\\s+.*"
        AddIfMissing "/etc/opendkim/opendkim.conf" "^InternalHosts\\s+refile:/etc/opendkim/trusted.hosts" "InternalHosts           refile:/etc/opendkim/trusted.hosts" "^InternalHosts\\s+.*"
        AddIfMissing "/etc/opendkim/opendkim.conf" "^ExternalIgnoreList\\s+refile:/etc/opendkim/trusted.hosts" "ExternalIgnoreList      refile:/etc/opendkim/trusted.hosts" "^ExternalIgnoreList\\s+.*"

        if ! [ -f "/etc/opendkim/trusted.hosts" ]; then
                echo "127.0.0.1" >> /etc/opendkim/trusted.hosts
        fi

        systemctl restart opendkim
fi

if [[ $CFG_POSTGRES_PASS -eq 1 ]]; then
        # Change the postgres user password
        pushd . >/dev/null
        cd /tmp
        sudo -u postgres psql -c "ALTER USER postgres WITH PASSWORD '$POSTGRES_PASSWORD'"
        popd >/dev/null
fi

if [[ $CFG_POSTGRES -eq 1 ]]; then
        # If there is no entry for md5 authentication in the pg_hba file for 127.0.0.1, add it
        grep -iEq "^host\s+all\s+all\s+127.0.0.1/32\s+md5" /var/lib/pgsql/data/pg_hba.conf
        if [[ $? -ne 0 ]]; then

                sed -i -E "s/(^host\\s+all\\s+all\\s+127.0.0.1\/32\\s+).*/\\1md5/i" /var/lib/pgsql/data/pg_hba.conf

        fi

        systemctl restart postgresql
fi

if [[ $CFG_POSTFIX -eq 1 ]]; then

        # Stop Postfix if it is running
        systemctl stop postfix
        systemctl disable postfix
fi

if [[ $CFG_FIREWALL -eq 1 ]]; then
        # Configure the firewall
        firewall-cmd --zone=public --permanent --add-service=http
        firewall-cmd --zone=public --permanent --add-service=https
        firewall-cmd --zone=public --permanent --add-service=smtp
        firewall-cmd --zone=public --permanent --add-service=smtps
        firewall-cmd --zone=public --permanent --add-service=postgresql
        firewall-cmd --zone=public --permanent --add-service=dns
        firewall-cmd --zone=public --permanent --add-port=4928/tcp
        firewall-cmd --zone=public --permanent --add-port=4929/tcp
        firewall-cmd --zone=public --permanent --add-port=4930/tcp
        firewall-cmd --zone=public --permanent --add-port=4932/tcp
        firewall-cmd --zone=public --permanent --add-port=3310/tcp
        firewall-cmd --reload
fi

echo ""
echo "Secure Messaging Gateway installation is finished."
echo ""
if [[ $CFG_SMG -eq 1 ]]; then
        echo "If you have any additional tasks necessary for your environment, make those"
        echo "changes now."
        echo ""
        if [[ $CFG_APACHE -eq 1 ]]; then
                echo "You can now connect to this server with a web browser to continue with the"
                echo "SMG specific setup process."
        else
                echo "Once you have configured your web server to host the SMG administration"
                echo "site, you can connect to the server with a web browser to continue with the"
                echo "SMG specific setup process."
        fi
fi
echo ""
