package SUSE_SMG::Parser::RepoList;
use strict;
use XML::Parser;
use SUSE_SMG::SRPrivate;

# The handler is called with something like this
#
# $VAR1 = {
#           'ALIAS'       => 'repo-oss',
#           'NAME'        => 'openSUSE-11.0-Oss',
#           'TYPE'        => 'yast2',
#           'ENABLED'     => 1,
#           'AUTOREFRESH' => 0,
#           'GPGCHECK'    => 1,
#           'URL'         => 'http://download.opensuse.org/distribution/11.0/repo/oss/'
#         };
#
# or
#
# $VAR1 = {
#           'ALIAS'       => 'SLES11-Updates',
#           'NAME'        => 'SLES11-Updates',
#           'TYPE'        => 'nu',
#           'ENABLED'     => 1,
#           'AUTOREFRESH' => 0,
#           'URL'         => 'https://nu.novell.com'
#           'CATALOGS'    => {
#                              SLES11-Updates => {
#                                                  'ALIAS'       => 'SLES11-Updates',
#                                                  'NAME'        => 'SLES11-Updates',
#                                                  'TYPE'        => 'yast2',
#                                                  'ENABLED'     => 1,
#                                                  'AUTOREFRESH' => 0,
#                                                  'GPGCHECK'    => 1,
#                                                  'URL'         => 'http://download.opensuse.org/distribution/11.0/repo/oss/'
#                                                }
#                             }
#         };



# constructor
sub new
{
    my $pkgname = shift;
    my %opt   = @_;
    my $self  = {};

    $self->{CURRENT}   = undef;
    $self->{HANDLER}   = undef;
    $self->{ELEMENT}   = undef;
    $self->{SERVICE}   = undef;
    $self->{CTX}       = undef;
    
    if(exists $opt{ctx} && defined $opt{ctx} && $opt{ctx})
    {
        $self->{CTX} = $opt{ctx};
    }
    bless($self);
    return $self;
}

# parses a xml resource
sub parse()
{
    my $self     = shift;
    my $xml      = shift;
    my $handler  = shift;

    $self->{HANDLER} = $handler;

    if (!defined $xml)
    {
        printLog($self->{CTX}, "error", "Invalid filename");
        return 1;
    }

    my $parser = XML::Parser->new( Handlers =>
                                   {
                                    Start=> sub { handle_start_tag($self, @_) },
                                    Char => sub { handle_char_tag($self, @_) },
                                    End=> sub { handle_end_tag($self, @_) },
                                   });

    eval
    {
        $parser->parse( $xml );
    };
    if ($@) {
        # ignore the errors, but print them
        chomp($@);
        SUSE_SMG::SRPrivate::printLog($self->{CTX}, "warn", "SUSE_SMG::Parser::RepoList Invalid XML \n$xml\n: $@");
    }
    return 0;
}

# handles XML reader start tag events
sub handle_start_tag()
{
    my $self = shift;
    my( $expat, $element, %attrs ) = @_;

    if(lc($element) eq "repo")
    {
        #$self->{SERVICE} = undef;
        $self->{ELEMENT} = "repo";
        $self->{CURRENT}->{URL} = "";
        
        if(exists $attrs{alias} && defined $attrs{alias})
        {
            $self->{CURRENT}->{ALIAS} = $attrs{alias};
        }
        if(exists $attrs{name} && defined $attrs{name})
        {
            $self->{CURRENT}->{NAME} = $attrs{name};
        }
        if(exists $attrs{type} && defined $attrs{type})
        {
            #
            # we need to "translate" the types to what we use internaly
            #
            if(lc($attrs{type}) eq "ris")
            {
                $self->{CURRENT}->{TYPE} = "nu";
            }
            elsif(lc($attrs{type}) eq "rpm-md")
            {
                $self->{CURRENT}->{TYPE} = "zypp";
            }
            elsif(lc($attrs{type}) eq "yast2")
            {
                $self->{CURRENT}->{TYPE} = "zypp";
            }
            else
            {
                $self->{CURRENT}->{TYPE} = lc($attrs{type});
            }
        }
        if(exists $attrs{enabled} && defined $attrs{enabled})
        {
            $self->{CURRENT}->{ENABLED} = int $attrs{enabled};
        }
        if(exists $attrs{autorefresh} && defined $attrs{autorefresh})
        {
            $self->{CURRENT}->{AUTOREFRESH} = int $attrs{autorefresh};
        }
        if(exists $attrs{gpgcheck} && defined $attrs{gpgcheck})
        {
            $self->{CURRENT}->{GPGCHECK} = int $attrs{gpgcheck};
        }
    }
    elsif(lc($element) eq "service")
    {
        $self->{SERVICE}->{URL} = "";
        $self->{ELEMENT} = "service";
        $self->{CURRENT} = undef;
        
        if(exists $attrs{alias} && defined $attrs{alias})
        {
            $self->{SERVICE}->{ALIAS} = $attrs{alias};
        }
        if(exists $attrs{name} && defined $attrs{name})
        {
            $self->{SERVICE}->{NAME} = $attrs{name};
        }
        if(exists $attrs{type} && defined $attrs{type})
        {
            #
            # we need to "translate" the types to what we use internaly
            #
            if(lc($attrs{type}) eq "ris")
            {
                $self->{SERVICE}->{TYPE} = "nu";
            }
            elsif(lc($attrs{type}) eq "rpm-md")
            {
                $self->{SERVICE}->{TYPE} = "zypp";
            }
            elsif(lc($attrs{type}) eq "yast2")
            {
                $self->{SERVICE}->{TYPE} = "zypp";
            }
            else
            {
                $self->{SERVICE}->{TYPE} = lc($attrs{type});
            }
        }
        else
        {
            # FIXME: workaround
            $self->{SERVICE}->{TYPE} = "nu";
        }
        if(exists $attrs{enabled} && defined $attrs{enabled})
        {
            $self->{SERVICE}->{ENABLED} = int $attrs{enabled};
        }
        if(exists $attrs{autorefresh} && defined $attrs{autorefresh})
        {
            $self->{SERVICE}->{AUTOREFRESH} = int $attrs{autorefresh};
        }
        if(exists $attrs{url} && defined $attrs{url})
        {
            $self->{SERVICE}->{URL} = $attrs{url};
        }
    }
    elsif(lc($element) eq "url")
    {
        $self->{ELEMENT} = "url";
    }
}

sub handle_char_tag
{
    my $self = shift;
    my( $expat, $string) = @_;

    chomp($string);
    return if($string =~ /^\s*$/);
    if($self->{ELEMENT} eq "url" && exists $self->{CURRENT}->{URL})
    {
        $self->{CURRENT}->{URL} .= $string;
    }
}

sub handle_end_tag
{
    my( $self, $expat, $element ) = @_;

    if(lc($element) eq "repo" && !defined $self->{SERVICE})
    {
        # first call the callback
        $self->{HANDLER}->($self->{CURRENT});

        $self->{ELEMENT} = undef;
        $self->{CURRENT} = undef;
    }
    elsif(lc($element) eq "repo" && defined $self->{SERVICE})
    {
        foreach my $key (keys %{$self->{CURRENT}})
        {
            $self->{SERVICE}->{CATALOGS}->{$self->{CURRENT}->{ALIAS}}->{$key} = $self->{CURRENT}->{$key};
        }
        $self->{ELEMENT} = undef;
        $self->{CURRENT} = undef;
    }
    elsif(lc($element) eq "service")
    {
        # first call the callback
        $self->{HANDLER}->($self->{SERVICE});
        
        $self->{SERVICE} = undef;
        $self->{ELEMENT} = undef;
        $self->{CURRENT} = undef;
    }
}

1;
