#
# Copyright (c) 2006 SUSE LINUX Products GmbH, Nuernberg, Germany.
# Copyright (c) 2011 SUSE LINUX Products GmbH, Nuernberg, Germany.
#


package SUSE_SMG::SuseRegister;

=head1 NAME

SUSE_SMG::SuseRegister - Functions for registration at the Novell registration server

To test for an error you have to look at the context error code $ctx->{errorcode}.
If the errorcode is not equal 0 an error happens. You can get a message via $ctx->{errormsg}.

=cut


use strict;
use XML::Parser;
use XML::Writer;
use Data::Dumper;
use Getopt::Long;
use Encode;
use Sys::Syslog;
use URI;
use URI::QueryParam;
use Time::HiRes qw(gettimeofday tv_interval);
use SUSE_SMG::SRPrivate;
use Fcntl qw(O_WRONLY O_CREAT O_APPEND);

=head2 Public Functions

The set of public functions

 use SUSE_SMG::SuseRegister;

=over 2

=item *
B<$ctx = init_ctx($data)>

Initialize the context. This function must be called first, before using the
rest of this functions.

I<$data> is a hash reference and supports the following keys:

* nooptional (bool)

* nohwdata (bool)

* forcereg (bool)

* xmlout (bool)

* logfile (filename)

* locale (string)

* noproxy (bool)

* yastcall (bool)

* norefresh (bool)

* debug (bool)

* args (hash reference with 'key' => 'value')

* extraCurlOption (array reference) *obsolete*

* time (bool)

* reqmirrcreds (bool)

* erase (bool)

For registration some arguments must be provided to the registration server.
These arguments are stored in side the context $ctx.
Here is the "$ctx->{args}" definition:

 $ctx->{args} => {
                   '<key>' => {
                                'value' => '<value>',
                                'flag'  => '<i|a|m>',
                                'kind'  => '<m|o>',
                                'description' => '...sometext...'
                              },
                    ...
                 };

 flag: i == initial   (still send to the server; cannot be discarded)
       a == automatic (automatic collected by the system; maybe discarded - see also "kind")
       m == manual    (must/should be set by the user - see also "kind")

 kind: m == mandatory (value must be set)
       o == optional  (value could be set)

 Every value given via init_ctx has the flag "i" and kind "m"
 After init_ctx only initial arguments are stored in this field. Every "needinfo"
 during the registration request( I<register($ctx)> ) can modify the arguments
 in this field.

EXAMPLE:

  my $data = {};
  $data->{nooptional} = 1;
  $data->{noproxy} = 1;

  my $ctx = SUSE_SMG::SuseRegister::init_ctx($data);
  if($ctx->{errorcode} != 0)
  {
    print STDERR $ctx->{errormsg}."\n";
    exit $ctx->{errorcode};
  }


=cut


sub init_ctx
{
    my $data = shift;
    my $ctx = {};
    my $code = 0;
    my $msg = "";

    $ctx->{errorcode} = 0;
    $ctx->{errormsg} = "";
    $ctx->{time} = 0;
    $ctx->{timeindent} = 0;
    $ctx->{logfile}     = undef;
    $ctx->{debug}       = 0;
    $ctx->{yastcall}    = 0; # FIXME: is this still needed?
    $ctx->{ncctranslate} = 0;
    $ctx->{LOGDESCR}    = undef;
    $ctx->{programStartTime} = [gettimeofday];
    $ctx->{printnodebug} = 0;
    $ctx->{reqmirrcreds} = 0;
    $ctx->{erase} = 0;

    # required to read informations with zypper even if another
    # instance of libzypp is running.
    $ENV{ZYPP_READONLY_HACK} = 1;

    if(exists $data->{debug} && defined $data->{debug})
    {
        $ctx->{debug} = $data->{debug};
    }

    if(exists $data->{printnodebug} && defined $data->{printnodebug})
    {
        $ctx->{printnodebug} = $data->{printnodebug};
    }

    if(exists $data->{logfile} && defined $data->{logfile})
    {
        $ctx->{logfile} = $data->{logfile};
    }
    elsif( exists $ENV{HOME} && defined $ENV{HOME} && $ENV{HOME} =~ /^\// )
    {
        # log by default to $HOME/.suse_register.log (bnc#552167)
        $ctx->{logfile} = $ENV{HOME}."/.suse_register.log";
    }

    if(exists $data->{yastcall} && defined $data->{yastcall})
    {
        $ctx->{yastcall} = $data->{yastcall};
    }

    if(exists $data->{ncctranslate} && defined $data->{ncctranslate})
    {
        $ctx->{ncctranslate} = $data->{ncctranslate};
    }

    if (defined $ctx->{logfile} && $ctx->{logfile} ne "")
    {
        # use 0600 for logfile permissions (bnc#725769)
        sysopen($ctx->{LOGDESCR}, $ctx->{logfile}, O_WRONLY|O_CREAT|O_APPEND, 0600) or do
        {
            if(!$ctx->{yastcall})
            {
                SUSE_SMG::SRPrivate::logPrintError($ctx, "Cannot open logfile <$ctx->{logfile}>: $!\n", 12);
                return $ctx;
            }
            else
            {
                syslog("err", "Cannot open logfile <$ctx->{logfile}>: $!(yastcall ignoring error)");
                $ctx->{LOGDESCR} = undef;
            }
        };
        # $LOGDESCR is undef if no logfile is defined
        if(defined $ctx->{LOGDESCR})
        {
            # autoflush ; removed - seems to have some side effects
            # $ctx->{LOGDESCR}->autoflush(1);
            print {$ctx->{LOGDESCR}} "----- ".localtime()." ---------------------------------------\n";
        }
    }


    if(exists $data->{time} && defined $data->{time})
    {
        $ctx->{time} = $data->{time};
    }
    my $t0 = [gettimeofday] if($ctx->{time});
    printLog($ctx, "info", SUSE_SMG::SRPrivate::indent($ctx)."START: init_ctx") if($ctx->{time});


    $ctx->{version} = "1.0";

    $ctx->{configFile}      = "/etc/suseRegister-smg.conf";
    $ctx->{sysconfigFile}   = "/etc/sysconfig/suse_register";

    # these are the old files.
    # if they still exists, we need to convert them
    # to the new format
    $ctx->{GUID_FILE}       = "/etc/zmd/deviceid";
    $ctx->{SECRET_FILE}     = "/etc/zmd/secret";

    # new credentails file
    $ctx->{CREDENTIAL_DIR}  = "/etc/zypp/credentials.d/";
    $ctx->{CREDENTIAL_FILE} = "NCCcredentials";

    $ctx->{SYSCONFIG_CLOCK} = "/etc/sysconfig/clock";
    $ctx->{CA_PATH}         = "/etc/ssl/certs";

    $ctx->{URL}             = "https://secure-www.novell.com/center/regsvc/";

    $ctx->{URLlistParams}   = "command=listparams";
    $ctx->{URLregister}     = "command=register";
    $ctx->{URLlistProducts} = "command=listproducts";

    $ctx->{guid}      = undef;
    $ctx->{secret}    = undef;
    $ctx->{locale}    = undef;
    $ctx->{encoding}  = "utf-8";
    $ctx->{lang}      = "en-US";

    $ctx->{listParams}  = 0;
    $ctx->{xmlout}      = 0;
    $ctx->{nooptional}  = 0;
    $ctx->{acceptmand}  = 0;
    $ctx->{forcereg}    = 0;
    $ctx->{nohwdata}    = 0;
    $ctx->{batch}       = 0;
    $ctx->{interactive} = 0;
    $ctx->{noproxy}     = 0;
    $ctx->{args}      = {
                         processor => { flag => "i", value => undef, kind => "mandatory" },
                         platform  => { flag => "i", value => undef, kind => "mandatory" },
                         timezone  => { flag => "i", value => undef, kind => "mandatory" },
#                         ostarget  => { flag => "i", value => undef, kind => "mandatory" },
    };
    $ctx->{serverKnownProducts} = [];
    $ctx->{installedProducts}   = [];
    $ctx->{products} = [];

    $ctx->{installedPatterns} = [];

    $ctx->{extraCurlOption} = [];

    $ctx->{hostGUID} = undef;
    $ctx->{virtType} = "";
    $ctx->{FallbackHostGUID} = undef;

    $ctx->{zmdConfig} = {};
#    $ctx->{ostarget} = "";

    $ctx->{norefresh} = 0;

    $ctx->{redirects} = 0;
    $ctx->{registerManuallyURL} = "";
    $ctx->{registerReadableText} = [];
    $ctx->{registerPrivPol} = "";

    $ctx->{registerStatus} = {};
    $ctx->{registerStatusFile} = "/var/lib/suseRegister/registration-status.xml";
    $ctx->{mirrorCredentialsFile} = "/var/lib/suseRegister/mirror-credentials.xml";

    $ctx->{zypper}        = "/usr/bin/zypper";
    $ctx->{lsb_release}   = "/usr/bin/lsb_release";
    $ctx->{uname}         = "/bin/uname";
    $ctx->{hwinfo}        = "/usr/sbin/hwinfo";
    $ctx->{curl}          = "/usr/bin/curl";

    $ctx->{xenstoreread}  = "/usr/bin/xenstore-read";
    $ctx->{xenstorewrite} = "/usr/bin/xenstore-write";
    $ctx->{xenstorechmod} = "/usr/bin/xenstore-chmod";
    $ctx->{lscpu}         = "/usr/bin/lscpu";

    $ctx->{createGuid}    = "/usr/bin/uuidgen";

    $ctx->{lastResponse}    = "";
    $ctx->{initialDomain}   = "";

    $ctx->{addRegSrvSrc} = 1;
    $ctx->{addAdSrc} = [];

    $ctx->{zmdcache} = "/var/cache/SuseRegister/lastzmdconfig.cache";
    $ctx->{restoreRepos} = 0;
    $ctx->{warnUnrestoredRepos} = 0;

    if(exists $data->{xmlout} && defined $data->{xmlout})
    {
        $ctx->{xmlout} = $data->{xmlout};
    }

    if(exists $data->{nooptional} && defined $data->{nooptional})
    {
        $ctx->{nooptional} = $data->{nooptional};
    }

    if(exists $data->{forcereg} && defined $data->{forcereg})
    {
        $ctx->{forcereg} = $data->{forcereg};
    }

    if(exists $data->{nohwdata} && defined $data->{nohwdata})
    {
        $ctx->{nohwdata} = $data->{nohwdata};
    }

    if(exists $data->{batch} && defined $data->{batch})
    {
        $ctx->{batch} = $data->{batch};
    }

    if(exists $data->{interactive} && defined $data->{interactive})
    {
        $ctx->{interactive} = $data->{interactive};
    }

    if(exists $data->{locale} && defined $data->{locale})
    {
        $ctx->{locale} = $data->{locale};
    }

    if(exists $data->{noproxy} && defined $data->{noproxy})
    {
        $ctx->{noproxy} = $data->{noproxy};
    }

    if(exists $data->{norefresh} && defined $data->{norefresh})
    {
        $ctx->{norefresh} = $data->{norefresh};
    }

    if(exists $data->{args} && ref($data->{args}) eq "HASH")
    {
        foreach my $a (keys %{$data->{args}})
        {
            $ctx->{args}->{$a} = {flag => "i", value => $data->{args}->{$a}, kind => "mandatory"};
        }
    }

    if(exists $data->{extraCurlOption} && ref($data->{extraCurlOption}) eq "ARRAY")
    {
        $ctx->{extraCurlOption} = $data->{extraCurlOption};
    }

    if(exists $data->{reqmirrcreds} && defined $ctx->{reqmirrcreds})
    {
        $ctx->{reqmirrcreds} = $data->{reqmirrcreds};
    }

    openlog("suse_register", "ndelay,pid", 'user');

    if(exists $data->{restoreRepos} && defined $data->{restoreRepos})
    {
        $ctx->{restoreRepos} = $data->{restoreRepos};
    }

    if(exists $ENV{LANG} && $ENV{LANG} =~ /^([\w_]+)\.?/)
    {
        if(defined $1 && $1 ne "")
        {
            $ctx->{lang} = $1;
            $ctx->{lang} =~ s/_/-/;
        }
    }
    elsif(exists $ENV{LANGUAGE} && $ENV{LANGUAGE} =~ /^([\w_]+)\.?/)
    {
        if(defined $1 && $1 ne "")
        {
            $ctx->{lang} = $1;
            $ctx->{lang} =~ s/_/-/;
        }
    }

    if (defined $ctx->{locale})
    {
        my ($l, $e) = split(/\.|@/, $ctx->{locale}, 2);

        if (defined $l && $l ne "")
        {
            $l =~ s/_/-/;
            $ctx->{lang} = $l;
        }

        if (defined $e && $e ne "")
        {
            $ctx->{encoding} = $e;
        }
    }

    # check for xen tools
    if(! -e $ctx->{xenstoreread} &&
       -e "/bin/xenstore-read")
    {
        $ctx->{xenstoreread} = "/bin/xenstore-read";
    }
    if(! -e $ctx->{xenstorewrite} &&
       -e "/bin/xenstore-write" )
    {
        $ctx->{xenstorewrite} = "/bin/xenstore-write";
    }
    if(! -e $ctx->{xenstorechmod} &&
       -e "/bin/xenstore-chmod" )
    {
        $ctx->{xenstorechmod} = "/bin/xenstore-chmod";
    }

    if(exists $data->{erase} && $data->{erase})
    {
        $ctx->{erase} = $data->{erase};
    }

    # call this as soon as possible.
    ($code, $msg) = SUSE_SMG::SRPrivate::initGUID($ctx);
    if($code != 0)
    {
        SUSE_SMG::SRPrivate::logPrintError($ctx, $msg, $code);
        return $ctx;
    }

    #
    # Call a refresh at the beginning, to be sure that we have a
    # complete an current state.
    #
    if(!$ctx->{norefresh})
    {
        my ($code, $message) = SUSE_SMG::SRPrivate::zyppRefresh($ctx);
        if($code != 0)
        {
            $message .= "\nRefresh failed. You may want to try again with --no-refesh option.";
            $ctx->{errorcode} = $code;
            $ctx->{errormsg}  = $message;
            return $ctx;
        }
    }

    ($code, $msg) = SUSE_SMG::SRPrivate::readSystemValues($ctx);
    if($code != 0)
    {
        SUSE_SMG::SRPrivate::logPrintError($ctx, $msg, $code);
        return $ctx;
    }


    printLog($ctx, "debug1", "list-parameters:   $ctx->{listParams}");
    printLog($ctx, "debug1", "xml-output:        $ctx->{xmlout}");
    printLog($ctx, "debug1", "no-optional:       $ctx->{nooptional}");
    printLog($ctx, "debug1", "batch:             $ctx->{batch}");
    printLog($ctx, "debug1", "forcereg:          $ctx->{forcereg}");
    printLog($ctx, "debug1", "no-hw-data:        $ctx->{nohwdata}");
    printLog($ctx, "debug1", "log:               ".(($ctx->{logfile})?$ctx->{logfile}:"undef"));
    printLog($ctx, "debug1", "locale:            ".(($ctx->{locale})?$ctx->{locale}:"undef"));
    printLog($ctx, "debug1", "no-proxy:          $ctx->{noproxy}");
    printLog($ctx, "debug1", "yastcall:          $ctx->{yastcall}");
    printLog($ctx, "debug1", "arg: ".Data::Dumper->Dump([$ctx->{args}]));
    printLog($ctx, "debug1", "extra-curl-option:".Data::Dumper->Dump([$ctx->{extraCurlOption}]));

    printLog($ctx, "debug1", "URL:               $ctx->{URL}");
    printLog($ctx, "debug1", "listParams:        $ctx->{URLlistParams}");
    printLog($ctx, "debug1", "register:          $ctx->{URLregister}");
    printLog($ctx, "debug1", "lang:              $ctx->{lang}");


    my $iuri = URI->new($ctx->{URL});

    $ctx->{initialDomain} = $iuri->host;
    $ctx->{initialDomain} =~ s/.+(\.[^.]+\.[^.]+)$/$1/;

    printLog($ctx, "debug1", "initialDomain:     $ctx->{initialDomain}");

    ($code, $msg) = SUSE_SMG::SRPrivate::listProducts($ctx);
    if($code != 0)
    {
        #SUSE_SMG::SRPrivate::logPrintError($ctx, $msg, $code);
        return $ctx;
    }

    $ctx->{products} = SUSE_SMG::SRPrivate::intersection($ctx, $ctx->{installedProducts}, $ctx->{serverKnownProducts});
    printLog($ctx, "debug1", "register products:".Data::Dumper->Dump([$ctx->{products}]));

    if($#{$ctx->{installedProducts}} == 0 &&
       exists $ctx->{installedProducts}->[0]->[0] &&
       $ctx->{installedProducts}->[0]->[0] =~ /FACTORY/i)
    {
        SUSE_SMG::SRPrivate::logPrintError($ctx, "FACTORY cannot be registered\n", 101);
        return $ctx;
    }

    if(@{$ctx->{products}} == 0)
    {
        SUSE_SMG::SRPrivate::logPrintError($ctx, "None of the installed products can be registered at the Novell registration server.\n", 100);
        return $ctx;
    }

    printLog($ctx, "info", SUSE_SMG::SRPrivate::indent($ctx)."END: init_ctx:".(tv_interval($t0))) if($ctx->{time});
    $ctx->{timeindent}--;

    return $ctx;
}

=item *
B<del_ctx($ctx)>

Delete the context

=cut

sub del_ctx
{
    my $ctx = shift;

    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    printLog($ctx, "info", SUSE_SMG::SRPrivate::indent($ctx)."START: del_ctx") if($ctx->{time});

    close $ctx->{LOGDESCR} if(defined $ctx->{LOGDESCR});
    closelog;

    $ctx = {};

    printLog($ctx, "info", SUSE_SMG::SRPrivate::indent($ctx)."END: del_ctx:".(tv_interval($t0))) if($ctx->{time});
    $ctx->{timeindent}--;
}


=item *
B<$txt = listParams($ctx)>

Return the parameter list from the server.

EXAMPLE:

  my $txt = listParams($ctx);
  if($ctx->{errorcode} != 0)
  {
    print STDERR $ctx->{errormsg}."\n";
    exit $ctx->{errorcode};
  }
  print $txt;

=cut

sub listParams
{
    my $ctx = shift;
    my $text = "";

    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    printLog($ctx, "info", SUSE_SMG::SRPrivate::indent($ctx)."START: listParams") if($ctx->{time});

    # cleanup the error status
    $ctx->{errorcode} = 0;
    $ctx->{errormsg} = "";

    my $output = '<?xml version="1.0" encoding="utf-8"?>';

    my $writer = new XML::Writer(OUTPUT => \$output);

    $writer->startTag("listparams", "xmlns" => "http://www.novell.com/xml/center/regsvc-1_0",
                                    "client_version" => "$SUSE_SMG::SRPrivate::SRversion");

    foreach my $PArray (@{$ctx->{products}})
    {
        if(defined $PArray->[0] && $PArray->[0] ne "" &&
           defined $PArray->[1] && $PArray->[1] ne "")
        {
            $writer->startTag("product",
                              "version" => $PArray->[1],
                              "release" => $PArray->[2],
                              "arch"    => $PArray->[3]);
            if ($PArray->[0] =~ /\s+/)
            {
                $writer->cdata($PArray->[0]);
            }
            else
            {
                $writer->characters($PArray->[0]);
            }
            $writer->endTag("product");
        }
    }

    $writer->endTag("listparams");

    printLog($ctx, "debug3", "XML:\n$output");

    $ctx->{redirects} = 0;
                                      # hard coded en-US; suse_register is in english
    my $res = SUSE_SMG::SRPrivate::sendData($ctx, $ctx->{URL}."?".$ctx->{URLlistParams}."&lang=en-US&version=$ctx->{version}", $output);
    if($ctx->{errorcode} != 0 || ! defined $res )
    {
        printLog($ctx, "info", "=>=>=> send Data failed => return");
        return;
    }

    if ($ctx->{xmlout})
    {
        return "$res\n";
    }
    else
    {
        my $privpol = "";

        if(!$ctx->{yastcall})
        {
            $text .= "Available Options:\n\n";
            $text .= "You can add these options to suse_register with the -a option.\n";
            $text .= "'-a' can be used multiple times\n\n";
        }

        my $p = new XML::Parser(Style => 'Objects', Pkg => 'SR');
        my $tree = $p->parse($res);

        #print Data::Dumper->Dump([$tree])."\n";

        if (! defined $tree || ref($tree->[0]) ne "SR::paramlist" ||
            ! exists $tree->[0]->{Kids} || ref($tree->[0]->{Kids}) ne "ARRAY")
        {
            SUSE_SMG::SRPrivate::logPrintError($ctx, "Invalid XML format. Cannot show human readable output. Try --xml-output.\n".
                                     6);
            return;
        }

        if($ctx->{yastcall})
        {
            $text .= "<pre>";
        }

        foreach my $kid (@{$tree->[0]->{Kids}})
        {
            #print Data::Dumper->Dump([$tree->[1]->[$i]])."\n\n";

            if (ref($kid) eq "SR::param")
            {
                if (exists $kid->{command} && defined $kid->{command} &&
                    $ctx->{nohwdata} && $kid->{command} =~ /^hwinfo/)
                {
                    # skip; --no-hw-data was provided
                    next;
                }
                elsif (exists $kid->{command} && defined $kid->{command} &&
                    $ctx->{nohwdata} && $kid->{command} =~ /^installed-desktops$/)
                {
                    # skip; --no-hw-data was provided
                    next;
                }

                $text .= "* ".$kid->{description}.": ";
                if(!$ctx->{yastcall})
                {
                    $text .= "\n\t".$kid->{id}."=<value> ";
                }

                if (exists $kid->{command} && defined $kid->{command} && $kid->{command} ne "")
                {
                    $text .= "(command: ".$kid->{command}.")\n";
                }
                else
                {
                    $text .= "\n";
                }
                if(!$ctx->{yastcall})
                {
                    $text .= "\n";
                }
            }
            elsif (ref($kid) eq "SR::privacy" )
            {
                if (exists $kid->{description} && defined $kid->{description})
                {
                    if(!$ctx->{yastcall})
                    {
                        $privpol .= "\nInformation on Novell's Privacy Policy:\n";
                        $privpol .= $kid->{description}."\n";
                    }
                    else
                    {
                        $privpol .= "<p>Information on Novell's Privacy Policy:<br>\n";
                        $privpol .= $kid->{description}."</p>\n";
                    }
                }

                if (exists $kid->{url} && defined $kid->{url} && $kid->{url} ne "")
                {
                    if(!$ctx->{yastcall})
                    {
                        $privpol .= $kid->{url}."\n";
                    }
                    else
                    {
                        $privpol .= "<p><a href=\"".$kid->{url}."\">";
                        $privpol .= $kid->{url}."</a></p>\n";
                    }
                }
            }
        }
        if(!$ctx->{yastcall})
        {
            $text .= "Example:\n";
            $text .= "  suse_register -a email=\"tux\@example.com\"\n";
        }
        else
        {
            $text .= "</pre>\n";
        }

        $text .= $privpol;
    }

    printLog($ctx, "info", SUSE_SMG::SRPrivate::indent($ctx)."END: listParams:".(tv_interval($t0))) if($ctx->{time});
    $ctx->{timeindent}--;

    return "$text\n";
}

=item *
B<$ret = register($ctx)>

Perform the registration.

I<$ret> returns the status of the registration.

 1 indicates a "needinfo" from the server.

 0 registration completed

 2 error - but it is better to test $ctx->{errorcode}

 3 error - mandatory hardware data required but --no-hw-data is given.

 4 needinfo which required manual interaction

Important values in the context are:

 $ctx->{args} the arguments which will be send to the server - see I<init_ctx()> for details.

 $ctx->{registerManuallyURL} contains a URL to a webpage where you can enter the manual values.

 $ctx->{registerReadableText} contains a human readable text about the missing arguments.


EXAMPLE:

  my $ret = register($ctx);
  if($ctx->{errorcode} != 0)
  {
    print STDERR $ctx->{errormsg}."\n";
    exit $ctx->{errorcode};
  }

  if($ret == 1 || $ret == 4)
  {
     # you can show and modify the $ctx->{args} here.
     # Have a look for mandatory args which cannot be automaticaly
     # detected. Ask for them, or open a browser with the URL proided
     # in $ctx->{registerManuallyURL} .

  }
  if($ret == 0)
  {
     # Registration completed. You can now get a tasklist to get
     # more information on the update sources (I<getTaskList($ctx)>).

  }

=cut


sub register
{
    my $ctx = shift;

    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    printLog($ctx, "info", SUSE_SMG::SRPrivate::indent($ctx)."START: register") if($ctx->{time});


    my $code = 0;
    my $msg = "";

    # cleanup the error status
    $ctx->{errorcode} = 0;
    $ctx->{errormsg} = "";

    my $output = SUSE_SMG::SRPrivate::buildXML($ctx);
    my $req_lang = "en-US";
    if ( $ctx->{ncctranslate} )
    {
        $req_lang = $ctx->{lang} if( defined $ctx->{lang} && $ctx->{lang} ne "");
    }

    $ctx->{redirects} = 0;
    my $res = SUSE_SMG::SRPrivate::sendData($ctx, $ctx->{URL}."?".$ctx->{URLregister}."&lang=$req_lang&version=$ctx->{version}", $output);
    if($ctx->{errorcode} != 0)
    {
      SUSE_SMG::SRPrivate::saveRegisterStatus($ctx, "");
      return 2;
    }

    if($res eq $ctx->{lastResponse})
    {
        # Got the same server response as the last time
        SUSE_SMG::SRPrivate::saveRegisterStatus($ctx, "");
        SUSE_SMG::SRPrivate::logPrintError($ctx, "Invalid response from the registration server. Aborting.\n", 9);
        return 2;
    }
    $ctx->{lastResponse} = $res;

    my $p = new XML::Parser(Style => 'Objects', Pkg => 'SR');
    my $tree = $p->parse($res);

    #print Data::Dumper->Dump([$tree])."\n";

    if (defined $tree && ref($tree->[0]) eq "SR::needinfo" &&
        exists $tree->[0]->{Kids} && ref($tree->[0]->{Kids}) eq "ARRAY")
    {
	# remove registration-status.xml content
	# we are in needinfo. we may get a new status later
        SUSE_SMG::SRPrivate::saveRegisterStatus($ctx, "");
        if ($ctx->{xmlout})
        {
            $ctx->{xmloutput} = "$res\n";
            return 1;
        }

        # cleanup old stuff
        $ctx->{registerReadableText} = [];
        $ctx->{registerManuallyURL} = "";
        $ctx->{registerPrivPol} = "";

        if (exists $tree->[0]->{href} &&
            defined $tree->[0]->{href} &&
            $tree->[0]->{href} ne "")
        {
            my $uri = URI->new($tree->[0]->{href});
            my $h = $uri->query_form_hash();
            $h->{lang} = "$ctx->{lang}";
            $uri->query_form_hash($h);
            $ctx->{registerManuallyURL} = $uri->as_string;
        }

        my $ret = SUSE_SMG::SRPrivate::evalNeedinfo($ctx, $tree);
        if($ctx->{errorcode} != 0)
        {
          return 3 if($ctx->{errorcode} == 3);
          return 2;
        }

        if(!$ret)
        {
            $ctx->{lastResponse} = "" if($ctx->{yastcall});
            $ret = 4;
        }

        if ($#{$ctx->{registerReadableText}} > -1 &&
            $ctx->{registerReadableText}->[$#{$ctx->{registerReadableText}}] =~ /^\s*$/)
        {
            pop @{$ctx->{registerReadableText}};
        }

        if(!$ctx->{yastcall})
        {
            unshift @{$ctx->{registerReadableText}},
            "To complete the registration, provide some additional parameters:\n\n";

            push @{$ctx->{registerReadableText}}, "\nYou can provide these parameters with the '-a' option.\n";
            push @{$ctx->{registerReadableText}}, "You can use the '-a' option multiple times.\n\n";
            push @{$ctx->{registerReadableText}}, "Example:\n\n";
            push @{$ctx->{registerReadableText}}, 'suse_register -a email="me@example.com"'."\n";
            push @{$ctx->{registerReadableText}}, "\nTo register your product manually, use the following URL:\n\n";
            push @{$ctx->{registerReadableText}}, "$ctx->{registerManuallyURL}\n\n";

        }
        else
        {
            unshift @{$ctx->{registerReadableText}}, "<pre>";
            push @{$ctx->{registerReadableText}}, "</pre>";
            push @{$ctx->{registerReadableText}}, "<p>To register your product manually, use the following URL:</p>\n";
            push @{$ctx->{registerReadableText}}, "<pre>$ctx->{registerManuallyURL}</pre>\n\n";
        }

        push @{$ctx->{registerReadableText}}, $ctx->{registerPrivPol};


        # after the first needinfo, set accept=mandatory to true
        # If the application think, that this is not a good idea
        # it can reset this.
        $ctx->{acceptmand} = 1;

        printLog($ctx, "info", SUSE_SMG::SRPrivate::indent($ctx)."END: register(needinfo):".(tv_interval($t0))) if($ctx->{time});
        $ctx->{timeindent}--;

        # return 1 == needinfo; 4 == needinfo, manual interaction required
        return $ret;
    }
    elsif (defined $tree && ref($tree->[0]) eq "SR::zmdconfig" &&
           exists $tree->[0]->{Kids} && ref($tree->[0]->{Kids}) eq "ARRAY")
    {
        $ctx->{newzmdconfig} = "$res";

        if ($ctx->{xmlout})
        {
            $ctx->{xmloutput} = "$res\n";
            return 0;
        }

        # ok => parse zmdconfig and add local configured repos from suseRegister-smg.conf
        ($code, $msg) = SUSE_SMG::SRPrivate::getZmdConfigValues($ctx);
        if($code != 0)
        {
            SUSE_SMG::SRPrivate::logPrintError($ctx, $msg, $code);
            return 2;
        }

        # read, what we get the last time
        ($code, $msg) = SUSE_SMG::SRPrivate::readLastZmdConfig($ctx);
        if($code != 0)
        {
            SUSE_SMG::SRPrivate::logPrintError($ctx, $msg, $code);
            return 2;
        }

        # read, what we have currently on the system
        ($code, $msg) = SUSE_SMG::SRPrivate::currentServices($ctx);
        if($code != 0)
        {
            SUSE_SMG::SRPrivate::logPrintError($ctx, $msg, $code);
            return 2;
        }
    }
    else
    {
      SUSE_SMG::SRPrivate::saveRegisterStatus($ctx, "");
      SUSE_SMG::SRPrivate::logPrintError($ctx, "Unknown reponse format.\n", 11);
      return 2;
    }
    printLog($ctx, "info", SUSE_SMG::SRPrivate::indent($ctx)."END: register(zmdconfig):".(tv_interval($t0))) if($ctx->{time});
    $ctx->{timeindent}--;
    return 0;
}

=item *
B<manageUpdateSources($ctx)>

Executes the update source configuration procedure. This method ignores all error which might
happen. When you test the context error code after executing this function you can see only
the last error or - if the errorcode is 0 - that no error happens.

If you want another behaviour fetch the tasklist (I<getTaskList()>) and develop your own function
to configure the sources.

EXAMPLE:

  manageUpdateSources($ctx);
  if($ctx->{errorcode} != 0)
  {
    print STDERR $ctx->{errormsg}."\n";
    exit $ctx->{errorcode};
  }
  exit 0; #finished

=cut

# was: configureZMD
sub manageUpdateSources
{
    my $ctx = shift;

    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    printLog($ctx, "info", SUSE_SMG::SRPrivate::indent($ctx)."START: manageUpdateSources") if($ctx->{time});

    my $lastcode = 0;
    my $lastmsg = "";

    # cleanup the error status
    $ctx->{errorcode} = 0;
    $ctx->{errormsg} = "";

    my $taskList = getTaskList($ctx);
    if($ctx->{errorcode} != 0)
    {
        return undef;
    }

    printLog($ctx, "debug2", "Work on Tasklist:\n".Data::Dumper->Dump([$taskList]));

    foreach my $service (keys %{$taskList})
    {
        printLog($ctx, "debug3", "Service '$service'");

        if($taskList->{$service}->{TASK} eq "a")
        {
            # add this service
            addService($ctx, $taskList->{$service});
            if($ctx->{errorcode} != 0)
            {
                $lastcode = $ctx->{errorcode};
                $lastmsg  = $ctx->{errormsg};
                $ctx->{errorcode} = 0;
                $ctx->{errormsg} = "";
            }
        }
        elsif($taskList->{$service}->{TASK} eq "d")
        {
            # delete this service
            deleteService($ctx, $taskList->{$service});
            if($ctx->{errorcode} != 0)
            {
                $lastcode = $ctx->{errorcode};
                $lastmsg  = $ctx->{errormsg};
                $ctx->{errorcode} = 0;
                $ctx->{errormsg} = "";
            }
        }
        elsif($taskList->{$service}->{TASK} eq "le" &&
              lc($taskList->{$service}->{TYPE}) eq "nu" &&
              exists $taskList->{$service}->{CATALOGS} &&
              ref($taskList->{$service}->{CATALOGS}) eq "HASH")
        {
            #
            # add and delete was handled before; leaveDisabled means that the
            # service is not present and should not be present. The only missing
            # task is leaveEnabled. We work on this here
            #

            # check for changes in the catalogs
            foreach my $name (keys %{$taskList->{$service}->{CATALOGS}})
            {
                printLog($ctx, "debug3", "Catalog '$name'");

                if($taskList->{$service}->{CATALOGS}->{$name}->{TASK} eq "a")
                {
                    # add catalog
                    enableCatalog($ctx, $service, $taskList->{$service}->{CATALOGS}->{$name});
                    if($ctx->{errorcode} != 0)
                    {
                        $lastcode = $ctx->{errorcode};
                        $lastmsg  = $ctx->{errormsg};
                        $ctx->{errorcode} = 0;
                        $ctx->{errormsg} = "";
                    }
                }
                elsif($taskList->{$service}->{CATALOGS}->{$name}->{TASK} eq "d")
                {
                    # delete catalog
                    disableCatalog($ctx, $service, $taskList->{$service}->{CATALOGS}->{$name});
                    if($ctx->{errorcode} != 0)
                    {
                        if(exists $ctx->{lastZmdConfig}->{$service}->{CATALOGS}->{$name} &&
                           ! exists $ctx->{zmdConfig}->{$service}->{CATALOGS}->{$name} &&
                           ! exists $ctx->{currentSources}->{$service}->{CATALOGS}->{$name})
                        {
                            # maybe the catalog is really not there; we do not set an error here
                            $ctx->{errorcode} = 0;
                            $ctx->{errormsg} = "";
                        }
                        else
                        {
                            $lastcode = $ctx->{errorcode};
                            $lastmsg  = $ctx->{errormsg};
                            $ctx->{errorcode} = 0;
                            $ctx->{errormsg} = "";
                        }
                    }
                }
                # else do nothing
            }
        }
        # else do nothing
    }

    if($lastcode == 0)
    {
        # zmd successfully configured - save the cache file
        SUSE_SMG::SRPrivate::saveLastZmdConfig($ctx);
    }

    SUSE_SMG::SRPrivate::logPrintError($ctx, $lastmsg, $lastcode);

    printLog($ctx, "info", SUSE_SMG::SRPrivate::indent($ctx)."END: manageUpdateSources:".(tv_interval($t0))) if($ctx->{time});
    $ctx->{timeindent}--;

    return undef;
}



=item *
B<$taskList = getTaskList($ctx)>

Evaluate whichs task should be done.

In the tasklist you can see all sources which exists in the past, current and will be available in the future
with a suggestion what to do with it. Therefor we have some task flags:

 * a  == add this source
 * d  == delete this source
 * le == leave untouched (source is enabled)
 * ld == leave untouched (source is disabled)

The structure of this tasklist looks like this:

$taskList = {
          'ftp://download.example.com/SLES-10-GMC/i386/DVD1?alias=SUSE-Linux-Enterprise-Server' => {
                                'url' => 'ftp://download.example.com/SLES-10-GMC/i386/DVD1?alias=SUSE-Linux-Enterprise-Server',
                                'task' => 'le',
                                'type' => 'ZYPP',
                                'alias' => 'SUSE-Linux-Enterprise-Server'
                                                                                                   },
          'https://update.novell.com' => {
                                'catalog' => {
                                        'SLES10-Updates' => {
                                             'oldurl' => 'https://update.novell.com/repo/$RCE/SLES10-Updates/sles-10-i686/',
                                             'url' => 'https://update.novell.com/repo/$RCE/SLES10-Updates/sles-10-i586/',
                                             'task' => 'r'
                                                            },
                                        'SLE10-Debuginfo-Updates' => {
                                             'url' => 'https://update.novell.com/repo/$RCE/SLE10-Debuginfo-Updates/sles-10-i586/',
                                             'task' => 'le'
                                                                     }
                                              },
                                'url' => 'https://update.novell.com',
                                'task' => 'le',
                                'type' => 'nu',
                                'alias' => 'update_novell_com'
                                        }
           };

=cut

sub getTaskList
{
    my $ctx = shift;

    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    printLog($ctx, "info", SUSE_SMG::SRPrivate::indent($ctx)."START: getTaskList") if($ctx->{time});

    my $taskList = {};
    my @allServices = ();
    my $service = undef;
    my $code = 0;
    my $msg = "";

    # cleanup the error status
    $ctx->{errorcode} = 0;
    $ctx->{errormsg} = "";

    printLog($ctx, "debug3", "getTaskList called");

    # call currentServices if they are not set in the context
    if(!exists $ctx->{currentSources} ||
       ref($ctx->{currentSources}) ne "HASH")
    {
        # to be able to run testcases we cannot do this always
        ($code, $msg) = SUSE_SMG::SRPrivate::currentServices($ctx);
        if($code != 0)
        {
            SUSE_SMG::SRPrivate::logPrintError($ctx, $msg, $code);
            return undef;
        }
    }

    printLog($ctx, "debug3", "---------------- lastZmdConfig --------------------");
    printLog($ctx, "debug3", Data::Dumper->Dump([$ctx->{lastZmdConfig}]));
    printLog($ctx, "debug3", "---------------- zmdConfig --------------------");
    printLog($ctx, "debug3", Data::Dumper->Dump([$ctx->{zmdConfig}]));
    printLog($ctx, "debug3", "---------------- currentSources --------------------");
    printLog($ctx, "debug3", Data::Dumper->Dump([$ctx->{currentSources}]));


    push @allServices, (keys %{$ctx->{lastZmdConfig}});
    push @allServices, (keys %{$ctx->{zmdConfig}});
    push @allServices, (keys %{$ctx->{currentSources}});

    # unique
    @allServices = keys %{{map {$_,1} @allServices}};


    # check all cases of the services

    foreach $service (@allServices)
    {
        #
        #compare only lastZmdConfig and zmdConfig
        #

        if(exists $ctx->{lastZmdConfig}->{$service} &&
           ! exists $ctx->{zmdConfig}->{$service} )
        {
            # delete it
            printLog($ctx, "debug3", "$service exists only in lastZmdConfig: delete it");
            $taskList->{$service} = {};
            SUSE_SMG::SRPrivate::copyService($taskList->{$service}, $ctx->{lastZmdConfig}->{$service});
            $taskList->{$service}->{TASK} = "d";
        }
        elsif(! exists $ctx->{lastZmdConfig}->{$service} &&
              exists $ctx->{zmdConfig}->{$service} )
        {
            # add it
            printLog($ctx, "debug3", "$service exists only in zmdConfig: add it");
            $taskList->{$service} = {};
            SUSE_SMG::SRPrivate::copyService($taskList->{$service}, $ctx->{zmdConfig}->{$service});
            if(! exists $ctx->{currentSources}->{$service})
            {
                $taskList->{$service}->{TASK} = "a";
            }
            else
            {
                $taskList->{$service}->{TASK} = "le";
            }
        }
        else
        {
            printLog($ctx, "debug3", "$service exists lastZmdConfig and zmdConfig or only in currentSources: leave enabled");
            $taskList->{$service} = {};
            if(! exists $ctx->{currentSources}->{$service})
            {
                SUSE_SMG::SRPrivate::copyService($taskList->{$service}, $ctx->{zmdConfig}->{$service});
                if($ctx->{restoreRepos})
                {
                    $taskList->{$service}->{TASK} = "a";
                }
                else
                {
                    $taskList->{$service}->{TASK} = "ld";
                    $ctx->{warnUnrestoredRepos} = 1;
                }
            }
            else
            {
                SUSE_SMG::SRPrivate::copyService($taskList->{$service}, $ctx->{currentSources}->{$service});
                $taskList->{$service}->{TASK} = "le";
            }
        }
    }

    # check catalogs for not yum/zypp service types

    foreach $service (keys %{$taskList})
    {
        printLog($ctx, "debug3", "search for special types in service '$service'");

        # "a" - it is not in currentSources and the catalog values are from zmdConfig
        # "d" - we want to delete it and the catalog values are from lastZmdConfig
        # "ld" - it is disabled
        # "le" - it is enabled

        if(lc($taskList->{$service}->{TYPE}) eq "nu" &&
           ($taskList->{$service}->{TASK} eq "a" || $taskList->{$service}->{TASK} eq "le") )
        {
            printLog($ctx, "debug3", "Found nu type with catalogs");

            if(! exists $ctx->{lastZmdConfig}->{$service} &&
               ! exists $ctx->{zmdConfig}->{$service} &&
               exists $ctx->{currentSources}->{$service})
            {
                # do nothing
                printLog($ctx, "debug3", "catalog only in currentSources - do nothing");

                # copy the subscribed tag to task
                foreach my $name (keys %{$taskList->{$service}->{CATALOGS}})
                {
                    my $task = ($taskList->{$service}->{CATALOGS}->{$name}->{ENABLED} == 1)?"le":"ld";
                    $taskList->{$service}->{CATALOGS}->{$name}->{TASK} = $task;
                }
            }
            else
            {
                printLog($ctx, "debug3", "catalog in all sources");

                my @allCatalogNames = ();
                push @allCatalogNames, (keys %{$ctx->{lastZmdConfig}->{$service}->{CATALOGS}});
                push @allCatalogNames, (keys %{$ctx->{zmdConfig}->{$service}->{CATALOGS}});
                push @allCatalogNames, (keys %{$ctx->{currentSources}->{$service}->{CATALOGS}});

                # unique
                @allCatalogNames = keys %{{map {$_,1} @allCatalogNames}};

                printLog($ctx, "debug3", "unique catalog count: ".@allCatalogNames);


                # first we delete the catalog entry of the current $taskList
                # and add an empty one
                #delete $taskList->{$service}->{CATALOGS};
                $taskList->{$service}->{CATALOGS} = {};

                # go through all cases
                foreach my $name (@allCatalogNames)
                {
                    printLog($ctx, "debug3", "check catalog '$name'");

                    #
                    # compare only lastZmdConfig and zmdConfig
                    #

                    if(exists $ctx->{lastZmdConfig}->{$service}->{CATALOGS}->{$name} &&
                       ! exists $ctx->{zmdConfig}->{$service}->{CATALOGS}->{$name} )
                    {
                        # we always want to delete the catalog. It may happen, that it disapeared
                        # during the refresh at the start of suse_register and we to remove it
                        # from the -to-enable list .
                        # bnc#570695, bnc#
                        printLog($ctx, "debug3", "delete catalog");
                        $taskList->{$service}->{CATALOGS}->{$name} = {};
                        SUSE_SMG::SRPrivate::copyCatalog($taskList->{$service}->{CATALOGS}->{$name},
                                                     $ctx->{lastZmdConfig}->{$service}->{CATALOGS}->{$name});
                        $taskList->{$service}->{CATALOGS}->{$name}->{TASK} = "d";
                    }
                    elsif(! exists $ctx->{lastZmdConfig}->{$service}->{CATALOGS}->{$name} &&
                          exists $ctx->{zmdConfig}->{$service}->{CATALOGS}->{$name} )
                    {
                        printLog($ctx, "debug3", "add catalog");
                        $taskList->{$service}->{CATALOGS}->{$name} = {};
                        SUSE_SMG::SRPrivate::copyCatalog($taskList->{$service}->{CATALOGS}->{$name},
                                                     $ctx->{zmdConfig}->{$service}->{CATALOGS}->{$name});
                        $taskList->{$service}->{CATALOGS}->{$name}->{TASK} = "a";
                    }
                    elsif(exists $ctx->{lastZmdConfig}->{$service}->{CATALOGS}->{$name} &&
                          exists $ctx->{zmdConfig}->{$service}->{CATALOGS}->{$name} )
                    {
                        $taskList->{$service}->{CATALOGS}->{$name} = {};
                        SUSE_SMG::SRPrivate::copyCatalog($taskList->{$service}->{CATALOGS}->{$name},
                                                     $ctx->{zmdConfig}->{$service}->{CATALOGS}->{$name});

                        # check enabled status
                        if(exists $ctx->{currentSources}->{$service}->{CATALOGS}->{$name}->{ENABLED} &&
                           $ctx->{currentSources}->{$service}->{CATALOGS}->{$name}->{ENABLED})
                        {
                            printLog($ctx, "debug3", "catalog leave enabled");
                            $taskList->{$service}->{CATALOGS}->{$name}->{TASK} = "le";
                        }
                        else
                        {
                            if($ctx->{restoreRepos})
                            {
                                printLog($ctx, "debug3", "add catalog");
                                $taskList->{$service}->{CATALOGS}->{$name}->{TASK} = "a";
                            }
                            else
                            {
                                printLog($ctx, "debug3", "catalog leave disabled");
                                $taskList->{$service}->{CATALOGS}->{$name}->{TASK} = "ld";
                                $ctx->{warnUnrestoredRepos} = 1;
                            }
                        }
                    }
                    elsif(exists $ctx->{currentSources}->{$service}->{CATALOGS}->{$name})
                    {
                        # This code should never been reached, but we implement it anyway.

                        $taskList->{$service}->{CATALOGS}->{$name} = {};
                        SUSE_SMG::SRPrivate::copyCatalog($taskList->{$service}->{CATALOGS}->{$name},
                                                     $ctx->{currentSources}->{$service}->{CATALOGS}->{$name});

                        my $task = ($taskList->{$service}->{CATALOGS}->{$name}->{ENABLED} == 1)?"le":"ld";
                        $taskList->{$service}->{CATALOGS}->{$name}->{TASK} = $task;
                    }
                }
            }
        }
    }

    printLog($ctx, "info", SUSE_SMG::SRPrivate::indent($ctx)."END: getTaskList:".(tv_interval($t0))) if($ctx->{time});
    $ctx->{timeindent}--;

    return $taskList;
}

sub saveLastZmdConfig
{
    my $ctx = shift;

    # zmd successfully configured - save the cache file
    return SUSE_SMG::SRPrivate::saveLastZmdConfig($ctx);
}



=item *
B<$ret = addService($ctx, $service)>

Add a Service. Reports only the last error.

EXAMPLE:


  my $ret = addService($ctx, $taskList->{http://update.novell.com/});
  if($ctx->{errorcode} != 0)
  {
    print STDERR $ctx->{errormsg}."\n";
    exit $ctx->{errorcode};
  }


=cut

sub addService
{
    my $ctx     = shift;
    my $service = shift || undef;

    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    printLog($ctx, "info", SUSE_SMG::SRPrivate::indent($ctx)."START: addService") if($ctx->{time});

    my $lastmsg     = "";
    my $lastcode    = 0;
    my $msg     = "";
    my $code    = 0;
    my $url     = "";
    my $type    = undef;
    my $alias   = "";

    # cleanup the error status
    $ctx->{errorcode} = 0;
    $ctx->{errormsg} = "";

    $url     = $service->{URL}     if(exists $service->{URL}) ;
    $type    = $service->{TYPE}    if(exists $service->{TYPE});
    $alias   = $service->{ALIAS}   if(exists $service->{ALIAS});

    if (! defined $url || $url eq "")
    {
        SUSE_SMG::SRPrivate::logPrintError($ctx, "Missing URL.\n", 14);
        return 14;
    }
    if (! defined $type || $type eq "")
    {
        SUSE_SMG::SRPrivate::logPrintError($ctx, "Missing Service Type.\n", 14);
        return 14;
    }

    printLog($ctx, "debug3", "addService '$url => $alias'");

    ($code, $msg) = SUSE_SMG::SRPrivate::zyppServiceAdd($ctx, $url, $type, $alias);
    if($code != 0)
    {
        $lastcode = $code;
        $lastmsg  = $msg;
    }

    if(lc($type) eq "nu" &&
       exists $service->{CATALOGS} &&
       ref($service->{CATALOGS}) eq "HASH")
    {
        foreach my $name (keys %{$service->{CATALOGS}})
        {
            enableCatalog($ctx, $alias, $service->{CATALOGS}->{$name});
            if($ctx->{errorcode} != 0)
            {
                $lastcode = $ctx->{errorcode};
                $lastmsg  = $ctx->{errormsg};
                $ctx->{errorcode} = 0;
                $ctx->{errormsg} = "";
            }
        }
    }

    SUSE_SMG::SRPrivate::logPrintError($ctx, $msg, $code);

    printLog($ctx, "info", SUSE_SMG::SRPrivate::indent($ctx)."END: addService:".(tv_interval($t0))) if($ctx->{time});
    $ctx->{timeindent}--;

    return $code;
}

=item *
B<$ret = enableCatalog($ctx, $service, $catalog)>

Enable a catalog in a specific service. Reports only the last error.

EXAMPLE:


  my $ret = enableCatalog($ctx, "http://update.novell.com/",
                          $taskList->{http://update.novell.com/}->{CATALOGS}->{SLES10-Updates});
  if($ctx->{errorcode} != 0)
  {
    print STDERR $ctx->{errormsg}."\n";
    exit $ctx->{errorcode};
  }


=cut

sub enableCatalog
{
    my $ctx     = shift;
    my $service = shift || undef;
    my $catalog = shift || undef;

    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    printLog($ctx, "info", SUSE_SMG::SRPrivate::indent($ctx)."START: enableCatalog") if($ctx->{time});

    my $lastmsg     = "";
    my $lastcode    = 0;
    my $msg     = "";
    my $code    = 0;

    # cleanup the error status
    $ctx->{errorcode} = 0;
    $ctx->{errormsg} = "";

    if(!defined $service || $service eq "" || ! defined $catalog || ref($catalog) ne "HASH")
    {
        SUSE_SMG::SRPrivate::logPrintError($ctx, "Invalid Catalog", 14);
        return 14;
    }

    printLog($ctx, "debug3", "enableCatalog '$catalog->{ALIAS}' in service '$service'");

    ($code, $msg) = SUSE_SMG::SRPrivate::zyppCatalogEnable($ctx, $service, $catalog);
    if($code != 0)
    {
        $lastcode = $code;
        $lastmsg = $msg;
    }

    SUSE_SMG::SRPrivate::logPrintError($ctx, $lastmsg, $lastcode);

    printLog($ctx, "info", SUSE_SMG::SRPrivate::indent($ctx)."END: enableCatalog:".(tv_interval($t0))) if($ctx->{time});
    $ctx->{timeindent}--;

    return 14;
}


=item *
B<$ret = deleteService($ctx, $service)>

Delete a Service. Reports only the last error.

EXAMPLE:


  my $ret = deleteService($ctx, $taskList->{http://update.novell.com/});
  if($ctx->{errorcode} != 0)
  {
    print STDERR $ctx->{errormsg}."\n";
    exit $ctx->{errorcode};
  }


=cut

sub deleteService
{
    my $ctx = shift;
    my $service = shift || undef;

    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    printLog($ctx, "info", SUSE_SMG::SRPrivate::indent($ctx)."START: deleteService") if($ctx->{time});

    my $lastmsg     = "";
    my $lastcode    = 0;

    my $msg     = "";
    my $code    = 0;
    my $url     = "";
    my $type    = undef;
    my $alias   = "";

    # cleanup the error status
    $ctx->{errorcode} = 0;
    $ctx->{errormsg} = "";

    $url     = $service->{URL}     if(exists $service->{URL}) ;
    $type    = $service->{TYPE}    if(exists $service->{TYPE});
    $alias   = $service->{ALIAS}   if(exists $service->{ALIAS});

    if (! defined $url || $url eq "")
    {
        SUSE_SMG::SRPrivate::logPrintError($ctx, "Missing URL.\n", 14);
        return 14;
    }
    if (! defined $type || $type eq "")
    {
        SUSE_SMG::SRPrivate::logPrintError($ctx, "Missing Service Type.\n", 14);
        return 14;
    }

    printLog($ctx, "debug3", "deleteService '$url => $alias'");

    ($code, $msg) = SUSE_SMG::SRPrivate::zyppServiceDelete($ctx, $alias, $type);
    if($code != 0 )
    {
        # try with the URL
        ($code, $msg) = SUSE_SMG::SRPrivate::zyppServiceDelete($ctx, $url, $type);

        # code 32 is "not found" which can be ignored on delete
        if($code != 0 && $code != 32)
        {
            $lastcode = $code;
            $lastmsg = $msg;
        }
    }

    SUSE_SMG::SRPrivate::logPrintError($ctx, $lastmsg, $lastcode);

    printLog($ctx, "info", SUSE_SMG::SRPrivate::indent($ctx)."END: deleteService:".(tv_interval($t0))) if($ctx->{time});
    $ctx->{timeindent}--;

    return $lastcode;
}


=item *
B<$ret = disableCatalog($ctx, $catalog)>

Disable a Catalog in the specified service. Reports only the last error.

EXAMPLE:


  my $ret = disableCatalog($ctx, "http://update.novell.com/",
                           $taskList->{http://update.novell.com/}->{CATALOGS}->{SLES10-Updates});
  if($ctx->{errorcode} != 0)
  {
    print STDERR $ctx->{errormsg}."\n";
    exit $ctx->{errorcode};
  }


=cut

sub disableCatalog
{
    my $ctx     = shift;
    my $service = shift || undef;
    my $catalog = shift || undef;

    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    printLog($ctx, "info", SUSE_SMG::SRPrivate::indent($ctx)."START: disableCatalog") if($ctx->{time});

    my $lastmsg     = "";
    my $lastcode    = 0;
    my $msg     = "";
    my $code    = 0;

    # cleanup the error status
    $ctx->{errorcode} = 0;
    $ctx->{errormsg} = "";

    if(!defined $service || $service eq "" || ! defined $catalog || ref($catalog) ne "HASH")
    {
        return SUSE_SMG::SRPrivate::logPrintError($ctx, "Invalid Catalog", 14);
        return 14;
    }

    printLog($ctx, "debug3", "disableCatalog '$catalog->{ALIAS}' in service '$service'");

    ($code, $msg) = SUSE_SMG::SRPrivate::zyppCatalogDisable($ctx, $service, $catalog);
    if($code != 0)
    {
        $lastcode = $code;
        $lastmsg = $msg;
    }

    SUSE_SMG::SRPrivate::logPrintError($ctx, $lastmsg, $lastcode);

    printLog($ctx, "info", SUSE_SMG::SRPrivate::indent($ctx)."END: disableCatalog:".(tv_interval($t0))) if($ctx->{time});
    $ctx->{timeindent}--;

    return $lastcode;
}

sub eraseRegistrationData
{
    my $ctx = shift;
    my ($code, $msg) = SUSE_SMG::SRPrivate::eraseRegData($ctx);
    if($code != 0)
    {
        SUSE_SMG::SRPrivate::logPrintError($ctx, $msg, $code);
    }
    return $code;
}

#=======================================================================================================
#===== END OF PUBLIC FUNCTIONS =========================================================================
#=======================================================================================================


sub fullpathOf
{
    my $ctx = shift;
    my $program = shift || undef;

    if(!defined $program || $program eq "" || $program !~ /^[\w_-]+$/)
    {
        return undef;
    }

    my $fullpath = `which $program 2>/dev/null`;
    chomp($fullpath);
    printLog($ctx, "debug2", "Fullpath:$fullpath");

    if (defined $fullpath && $fullpath ne "")
    {
        return $fullpath;
    }
    return undef;
}

1;


